<?php

namespace App\Http\Controllers;

use App\Events\SendNewOrderReceived;
use App\Models\Order;
use App\Models\Payment;
use App\Models\PaymentGatewayCredential;
use App\Models\Restaurant;
use App\Models\RestaurantPayment;
use App\Models\StripePayment;
use App\Models\SuperadminPaymentGateway;
use App\Models\User;
use App\Notifications\NewOrderReceived;
use App\Scopes\BranchScope;
use Error;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Notification;

class StripeController extends Controller
{

    public function orderPayment()
    {
        $milestonePayment = StripePayment::findOrFail(request()->order_payment);
        $paymentGateway = $milestonePayment->order->branch->restaurant->paymentGateways;

        $stripe = new \Stripe\StripeClient($paymentGateway->stripe_secret);

        $checkoutSession = $stripe->checkout->sessions->create([
          'line_items' => [[
            'price_data' => [
              'currency' => 'usd',
              'product_data' => [
                'name' => 'Order #' .$milestonePayment->order->order_number,
              ],
              'unit_amount' => floatval($milestonePayment->amount * 100),
            ],
            'quantity' => 1,
          ]],
          'mode' => 'payment',
          'success_url' => route('stripe.success').'?session_id={CHECKOUT_SESSION_ID}',
          'cancel_url' => module_enabled('Subdomain') ? url('/') : route('shop_restaurant', ['hash' => $milestonePayment->order->branch->restaurant->hash]),
          'client_reference_id' => $milestonePayment->id
        ]);

        $milestonePayment->stripe_session_id = $checkoutSession->id;
        $milestonePayment->save();

        header('HTTP/1.1 303 See Other');
        header('Location: ' . $checkoutSession->url);
    }

    public function success()
    {
        $payment = StripePayment::where('stripe_session_id', request()->session_id)->firstOrFail();

        $paymentGateway = $payment->order->branch->restaurant->paymentGateways;

        $stripe = new \Stripe\StripeClient($paymentGateway->stripe_secret);

        try {
            $session = $stripe->checkout->sessions->retrieve(request()->session_id);

            $payment->stripe_payment_intent = $session->payment_intent;
            $payment->payment_status = 'completed';
            $payment->payment_date = now()->toDateTimeString();
            $payment->save();

            Payment::create([
                'order_id' => $payment->order_id,
                'payment_method' => 'stripe',
                'amount' => $payment->amount,
                'transaction_id' => $session->payment_intent
            ]);

            $order = Order::find($payment->order_id);
            $order->amount_paid = $payment->amount;
            $order->status = 'paid';
            $order->save();

            $this->sendNotifications($order);

            return redirect()->route('order_success', $payment->order_id);

        } catch (Error $e) {
            http_response_code(500);
            logger(json_encode(['error' => $e->getMessage()]));
        }

    }

    public function licensePayment()
    {
        $paymentGateway = SuperadminPaymentGateway::first();

        $milestonePayment = RestaurantPayment::findOrFail(request()->license_payment);

        $stripe = new \Stripe\StripeClient($paymentGateway->stripe_secret);

        $checkoutSession = $stripe->checkout->sessions->create([
          'line_items' => [[
            'price_data' => [
              'currency' => 'usd',
              'product_data' => [
                'name' => 'License Payment for ' . global_setting()->name,
              ],
              'unit_amount' => floatval($milestonePayment->amount * 100),
            ],
            'quantity' => 1,
          ]],
          'mode' => 'payment',
          'success_url' => route('stripe.license_success').'?session_id={CHECKOUT_SESSION_ID}',
          'cancel_url' => route('dashboard'),
          'client_reference_id' => $milestonePayment->id
        ]);

        $milestonePayment->stripe_session_id = $checkoutSession->id;
        $milestonePayment->save();

        header('HTTP/1.1 303 See Other');
        header('Location: ' . $checkoutSession->url);
    }

    public function licenseSuccess()
    {
        $paymentGateway = SuperadminPaymentGateway::first();

        $payment = RestaurantPayment::where('stripe_session_id', request()->session_id)->firstOrFail();
        $stripe = new \Stripe\StripeClient($paymentGateway->stripe_secret);

        try {
            $session = $stripe->checkout->sessions->retrieve(request()->session_id);

            $payment->stripe_payment_intent = $session->payment_intent;
            $payment->status = 'paid';
            $payment->payment_date_time = now()->toDateTimeString();
            $payment->save();
  
            Restaurant::where('id', $payment->restaurant_id)->update(['license_type' => 'paid']);

            session(['restaurant' => $payment->restaurant]);

            return redirect()->route('dashboard');


        } catch (Error $e) {
            http_response_code(500);
            logger(json_encode(['error' => $e->getMessage()]));
        }

    }

    public function sendNotifications($order)
    {       
        SendNewOrderReceived::dispatch($order);
    }

}
